-- Calcul morphologie PLL

------------------------------------------------------------------------------------------------------------------------
-- chenal
------------------------------------------------------------------------------------------------------------------------
-- paramètres : 
-- ref_schema => schéma contenant les données de référence (usra, tgh, bv )
-- col_schema => schéma contenant les données de collecte 
drop function if exists calc_chenal(ref_schema text, col_schema text);
create or replace function calc_chenal(ref_schema text, col_schema text)
    returns int
    language plpgsql
as
$calc_chenal$
DECLARE
    where_clause text;
    geom_col text; -- todo: from catalog ?
    row_cnt int8;
BEGIN
    -- query according to geo region:
    if ref_schema = 'ref_hexagone' then
        geom_col := 'geometrie';
        where_clause := $$nature in ('Canal', 'Ecoulement canalisé', 'Conduit buse')$$;
    else
        geom_col := 'geom';
        where_clause := $$nature in ('Canal, chenal', 'Canal')$$;
    end if;

    execute format ($$alter table %1$I.usra drop column if exists chenal;
      alter table %1$I.usra add column chenal text default 'faible';
      alter table %1$I.usra drop column if exists chenal_v;
      alter table %1$I.usra add column chenal_v double precision;
      comment on column %1$I.usra.chenal is 'taux de chenalisation du tronçon';
      comment on column %1$I.usra.chenal_v is 'taux de chenalisation du tronçon';

      with tmp as (
        select u.usra_id,
               st_length(
                       st_linemerge(
                               st_union(
                                       st_force2D(
                                               st_multi(
                                                       st_collectionExtract(
                                                               st_intersection(u.geom, c.%4$s, 3),
                                                               2)))))
               ) * 100 / st_length(u.geom) as ratio_canal
        from %1$I.usra u
                 join %2$I.troncon_hydrographique c on st_dwithin(u.geom, c.%4$s,1)
        where %3$s
        group by u.usra_id, u.geom
    ), tmp1 as (
        select usra_id, case
                    when ratio_canal < 20 then 'faible'
                    when ratio_canal >= 20 and ratio_canal < 50 then 'moyen'
                    else 'fort' end as chenal,ratio_canal as chenal_v
        from tmp
    )
    update %1$I.usra u set chenal_v= ROUND(tmp1.chenal_v::numeric,2), chenal = tmp1.chenal
    from tmp1
    where u.usra_id = tmp1.usra_id$$, ref_schema, col_schema, where_clause, geom_col);

    GET DIAGNOSTICS row_cnt = ROW_COUNT;
    raise notice 'Mise à jour de % usra pour la colonne chenal', row_cnt;
    return 0;
end
$calc_chenal$;

------------------------------------------------------------------------------------------------------------------------
-- txamg
------------------------------------------------------------------------------------------------------------------------
-- paramètres : 
-- trt_schema => schéma de traitement pour le stockage des tables temporaires
-- ref_schema => schéma contenant les données de référence (usra, tgh, bv )
-- col_schema => schéma contenant les données de collecte 
drop function if exists calc_txamg(trt_schema text, ref_schema text, col_schema text, bdtopo_schema text);
create or replace function calc_txamg(trt_schema text, ref_schema text, col_schema text, bdtopo_schema text)
    returns int
    language plpgsql
as
$calc_txamg$
DECLARE
    querystr text;
    srid int;
    row_cnt int8;
BEGIN
    RAISE NOTICE 'schéma trt (%)', trt_schema;
    RAISE NOTICE 'schéma ref (%)', ref_schema;
    RAISE NOTICE 'schéma collecte (%)', col_schema;
    RAISE NOTICE 'schéma bdtopo (%)', bdtopo_schema;

    -- gets srid from usra table:
    execute format($$select st_srid(geom) from %I.usra limit 1$$, ref_schema) into srid;
    RAISE NOTICE 'SRID pour la table amenagement: %', srid;
    RAISE NOTICE 'creation de la table amenagement...';

    -- table des amenagements
    execute format($$
        drop table if exists %1$I.amenagement;
        create table %1$I.amenagement (
            amenagement_id int8 generated always as identity primary key ,
            source_id text,
            source text,
            geom geometry(geometry, %2$L)
        );
    $$, trt_schema, srid);

    -- voies de communications ferres
    -- pas a mayotte
    if trt_schema <> 'trt_mayotte' then
        execute format($$
            insert into %1$I.amenagement(source_id, source, geom)
            select t.cleabs, 'troncon_de_voie_ferree', st_force2d(geometrie)
            from %2$I.troncon_de_voie_ferree t;
        $$,trt_schema, bdtopo_schema);
    end if;

    -- voies de communications: routes
    execute format($$
        insert into %1$I.amenagement(source_id, source, geom)
        select t.cleabs, 'troncon_de_route', st_force2d(geometrie)
        from %2$I.troncon_de_route t
        where nature in ('Bac ou liaison maritime', 'Bretelle', 'Escalier', 'Piste cyclable','Rond-point',
                         'Route à 1 chaussée', 'Route à 2 chaussées', 'Route empierrée', 'Type autoroutier');
    $$,trt_schema, bdtopo_schema);

    -- canaux et plans d'eau artificiels
    execute format($$
        insert into %1$I.amenagement(source_id, source, geom)
        select t.cleabs, 'surface_hydrographique', st_force2d(geometrie)
        from %2$I.surface_hydrographique t
        where nature in ('plan d''eau de gravière', 'Réservoir-bassin', 'Réservoir-bassin d''orage',
                         'Réservoir-bassin piscicole', 'Retenue-bassin portuaire', 'Canal', 'Ecoulement canalisé');
    $$,trt_schema, bdtopo_schema);

    -- murs et quais
    execute format($$
        insert into %1$I.amenagement(source_id, source, geom)
        select t.cleabs, 'construction_lineaire', st_force2d(geometrie)
        from %2$I.construction_lineaire t
        where nature in ('Quai', 'Mur', 'Mur de soutènement');
    $$,trt_schema, bdtopo_schema);

    -- terrains de sport
    execute format($$
        insert into %1$I.amenagement(source_id, source, geom)
        select t.cleabs, 'terrain_de_sport', st_force2d(geometrie)
        from %2$I.terrain_de_sport t;
    $$,trt_schema, bdtopo_schema);

    -- cimetières
    execute format($$
        insert into %1$I.amenagement(source_id, source, geom)
        select t.cleabs, 'cimetiere', st_force2d(geometrie)
        from %2$I.cimetiere t;
    $$,trt_schema, bdtopo_schema);

    -- aérodromes
    execute format($$
        insert into %1$I.amenagement(source_id, source, geom)
        select t.cleabs, 'aerodrome', st_force2d(geometrie)
        from %2$I.aerodrome t;
    $$,trt_schema, bdtopo_schema);

    -- autres équipements de transport
    execute format($$
        insert into %1$I.amenagement(source_id, source, geom)
        select t.cleabs, 'equipement_de_transport', st_force2d(geometrie)
        from %2$I.equipement_de_transport t;
    $$,trt_schema, bdtopo_schema);

    -- digues: custom suivant zones:
    if trt_schema = 'trt_martinique' then
        execute format($$
            insert into %1$I.amenagement(source_id, source, geom)
            select t.gid, 'digue', st_transform(geom, 5490)
            from ode972.digue t;
        $$,trt_schema);
    elsif trt_schema = 'trt_reunion' then
        execute format($$
        insert into %1$I.amenagement(source_id, source, geom)
        select t.gid, 'digue', geom
        from %2$I.digue t;
        $$,trt_schema, col_schema);
    end if;

    -- zones urbaines et périurbaines: custom suivant zones
    if trt_schema = 'trt_metro' then
        execute format($$
            insert into %1$I.amenagement(source_id, source, geom)
            select t.gid, 'oso2021', geom
            from %2$I.oso2021 t
            where classe < 5;
        $$,trt_schema, col_schema);
    elsif trt_schema = 'trt_guadeloupe' then
        execute format($$
            insert into %1$I.amenagement(source_id, source, geom)
            select t.ogc_fid, 'karucover_2017', geom
            from %2$I.karucover_2017 t
            where typcsn2 = 'CS1.1';
        $$,trt_schema, col_schema);

    elsif trt_schema = 'trt_martinique' then
        execute format($$
            insert into %1$I.amenagement(source_id, source, geom)
            select t.gid, 'occsol_2017', geom
            from %2$I.occsol_2017 t
            where CODE_CS in ('CS1.1.1.1', 'CS1.1.1.2', 'CS1.1.2.1', 'CS1.1.2.2')
               or (CODE_CS in ('CS2.2.1') and CODE_US in ('US235'));
        $$,trt_schema, col_schema);
    elsif trt_schema = 'trt_reunion' then
        execute format($$
            insert into %1$I.amenagement(source_id, source, geom)
            select t.gid, 'occsol', geom
            from %2$I.occsol t
            where code1 = 4;
        $$,trt_schema, col_schema);
    elsif trt_schema = 'trt_guyane' then
        execute format($$
            insert into %1$I.amenagement(source_id, source, geom)
            select t.gid, 'occsol_2015', geom
            from %2$I.occsol_2015 t
            where niveau3_15 in ('111', '112', '121', '122', '123', '132', '133', '14');
        $$,trt_schema, col_schema);
    elsif trt_schema = 'trt_mayotte' then
        execute format($$
            insert into %1$I.amenagement(source_id, source, geom)
            select t.gid, 'occsol_2016', geom
            from %2$I.occsol_2016 t
            where CODE_CS in ('CS1.1.1.1', 'CS1.1.1.2', 'CS1.1.2.1', 'CS1.1.2.2')
               or (CODE_CS in ('CS2.2.1') and CODE_US in ('US235'));
        $$,trt_schema, col_schema);
    end if;

    -- orpaillage: custom suivant zones
    if trt_schema = 'trt_guyane' then
        execute format($$
            insert into %1$I.amenagement(source_id, source, geom)
            select t.ogc_fid, 'col_guyane.camino', geom
            from %2$I.camino t;
        $$,trt_schema, col_schema);
    end if;

    execute format($$
        create index on %1$I.amenagement using gist(geom);
        analyse %1$I.amenagement;
    $$,trt_schema);

    -- subdivide now to work with small objects (occsol contains huge polygons)
    execute format($$
        drop table if exists %1$I.amenagement_sub;
        create table %1$I.amenagement_sub as
        select amenagement_id,
               st_subdivide(geom) as geom
        from %1$I.amenagement;
    $$,trt_schema);

    execute format($$
        create index on %1$I.amenagement_sub using gist(geom);
        analyse %1$I.amenagement_sub;
    $$,trt_schema);

    RAISE NOTICE 'table %.amenagement créée', trt_schema;
    RAISE NOTICE 'calcul des contraintes laterales...';

    -- calcul des contraintes
    querystr = format($$
        drop table if exists usra_amenagement_3w;
        create temporary table usra_amenagement_3w as
            select u.usra_id, (st_dump(st_intersection(u.geom, a.geom))).geom as geom
            from %1$I.usra_3w u join %2$I.amenagement_sub a on st_intersects(u.geom, a.geom);

        alter table usra_amenagement_3w add column id int generated always as identity primary key ;
        create index on usra_amenagement_3w  using gist(geom);
        analyse usra_amenagement_3w;

        -- line intercept: project amenagement points to each usra, keeping ids and locus to identify and build intercept segments
        drop table if exists usra_contrainte;
        create temporary table usra_contrainte as
            with tmp as (
                select ua.usra_id,
                    ua.id as amen_id,
                    st_linelocatepoint(u.geom, (d).geom) as locus,
                    u.geom
                from usra_amenagement_3w ua
                      cross join st_dumppoints(ua.geom) as d
                      join %1$I.usra u on ua.usra_id = u.usra_id
        --          where ua.usra_id = 220526
            ), tmp1 as (
                select tmp.usra_id, tmp.amen_id,
                       st_linesubstring(tmp.geom, min(locus), max(locus)) as geom
                from tmp
                group by tmp.usra_id, tmp.amen_id, tmp.geom
            )
            select usra_id, st_linemerge(st_union(geom, 0.01)) as geom -- introduce precision to circumvent union topo errors
                                                                      -- => lines not merged => overlapping lines produce wrong total length
            from tmp1
            group by usra_id;

        alter table usra_contrainte add primary key (usra_id);
        $$, ref_schema, trt_schema , col_schema);
    EXECUTE querystr;

    RAISE NOTICE 'contraintes laterales calculées.';
    RAISE NOTICE 'mise a jour de la table des usra, colonne txamg';

    --     1 faible taux d'aménagement à proximité du lit mineur faible < 10%
--     2 moyen taux d'aménagement à proximité du lit mineur faible >= 10%
--     3 fort taux d'aménagement à proximité du lit mineur moyen >= 25%
--     4 tres_fort taux d'aménagement à proximité du lit mineur très fort >= 50%
    execute format($$alter table %1$I.usra drop column if exists txamg ;
      alter table %1$I.usra add column txamg text default 'faible';
      alter table %1$I.usra drop column if exists txamg_v ;
      alter table %1$I.usra add column txamg_v double precision default 0;

      comment on column %1$I.usra.txamg is 'taux d''aménagements à proximité du lit mineur (3W)';
      comment on column %1$I.usra.txamg_v is 'taux d''aménagements à proximité du lit mineur (3W)';
      with tmp as (
        select u.usra_id, round((coalesce(st_length(uc.geom), 0) * 100 / st_length(u.geom))::numeric, 2) as tx_contrainte
        from %1$I.usra u join usra_contrainte uc using (usra_id)
      )
      update %1$I.usra a set txamg_v=uc.tx_contrainte,txamg = case when uc.tx_contrainte < 10 then 'faible'
                                         when uc.tx_contrainte >= 10 and uc.tx_contrainte < 25 then 'moyen'
                                         when uc.tx_contrainte >= 25 and uc.tx_contrainte < 50 then 'fort'
                                         when uc.tx_contrainte >= 50 then 'tres_fort' end
      from tmp uc
      where a.usra_id = uc.usra_id
    $$, ref_schema, trt_schema);
    GET DIAGNOSTICS row_cnt = ROW_COUNT;
    raise notice 'Mise à jour de % usra pour la colonne txamg', row_cnt;

    return 0;
END;
$calc_txamg$;


-- paramètres : 
-- ref_schema => schéma contenant les données de référence (usra, tgh, bv )
-- col_schema => schéma contenant les données de collecte 
create or replace function calc_navigable(ref_schema text,col_schema text)
    returns int
    language plpgsql
as
$calc_navigable$
DECLARE
    querystr text;
    where_clause text;
    geo_col text := 'geom';
    row_cnt int8;
BEGIN
    RAISE NOTICE 'schéma ref (%)', ref_schema;
    RAISE NOTICE 'schéma collecte (%)', col_schema;

    if ref_schema = 'ref_hexagone' then
        where_clause := 'd.navigabilite';
        geo_col := 'geometrie';
    elsif ref_schema in ('ref_guadeloupe', 'ref_martinique') then
        where_clause := $$d.navigabili = 'Navigable'$$;
    elsif ref_schema in ('ref_guyane', 'ref_reunion') then
        where_clause := $$d.navigable = 'Navigable'$$;
    else
        -- no navigable column in carthage mayotte
        where_clause := $$false$$;
    end if;

    querystr := format($$
      alter table %1$I.usra drop column if exists navigable ;
      alter table %1$I.usra add column navigable text default 'non';
      comment on column %1$I.usra.navigable is 'navigabilité du tronçon';

      With navi as (
        select u.usra_id,
               sum(st_length(st_intersection(u.geom, d.%4$I,1)))*100/st_length(u.geom) as ratio_inter
        from %1$I.usra u join %2$I.troncon_hydrographique d on st_dwithin(d.%4$I,u.geom,1)
        where %3$s
        group by u.usra_id, u.geom
    )
    update %1$I.usra u set navigable = 'oui'
    from navi n
    where u.usra_id = n.usra_id and ratio_inter >= 10;$$, ref_schema, col_schema, where_clause, geo_col);

    RAISE NOTICE '%',querystr;
    EXECUTE querystr;
    GET DIAGNOSTICS row_cnt = ROW_COUNT;
    raise notice 'Mise à jour de % usra pour la colonne ''navigable''', row_cnt;

    return 0;
END;
$calc_navigable$;
